<?php

namespace App\Http\Controllers\API\V1\Sales;
use App\Http\Controllers\API\V1\BaseController;
use Illuminate\Http\Request;

use App\Classes\Excel\InvoicesExport;
use App\Classes\General\Upload;
use App\Classes\General\General;
use App\Classes\Sales\ProformaInvoiceHelper;

use App\Models\Finance\Tax;
use App\Models\Inventory\Item;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\Branch;
use App\Models\Main\ScreenSub;
use App\Models\Main\Setting;
use App\Models\Sales\Customer;
use App\Models\Sales\ProformaInvoice;
use App\Models\Sales\Quotation;
use App\Models\Sales\SalesAgent;

use Salla\ZATCA\GenerateQrCode;
use Salla\ZATCA\Tags\InvoiceDate;
use Salla\ZATCA\Tags\InvoiceTaxAmount;
use Salla\ZATCA\Tags\InvoiceTotalAmount;
use Salla\ZATCA\Tags\Seller;
use Salla\ZATCA\Tags\TaxNumber;

use DB;
use Auth;
use Str;
use App;
use App\Models\Finance\Currency;
use PDF;
use Excel;

class ProformaInvoiceController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {

        $this->checkPermission('06-006','view');

        $this->filter = array(
            'lang'         => $lang ,
            'active'       => $request['active'] ?? 1 ,
            'status'       => $request['status'] ?? 1 ,
            'rows'         => $request['rows'] ?? 20 ,
            'page'         => $request['page'] ?? 1 ,
            'word'         => $request['word'] ?? null ,
            'branch'       => $this->data['current_branch'] ,
            'finance_year' => $this->data['current_year']
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-006', $lang)->sub_title ?? '';
        $this->return['items']       = ProformaInvoice::all_proforma_invoices($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {

        $this->checkPermission('06-006','add');

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-006', $lang)->sub_title ?? '';
        $this->return['quotations']        = Quotation::all_quotations(array('active' => 1, 'lang' => $lang));
        $this->return['customers']         = Customer::all_customers(array('active' => 1, 'lang' => $lang ,  'is_search' => 1));
        $this->return['sales_agents']      = SalesAgent::all_sales_agents(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang , 'is_search' => 1));
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['width_height']      = boolval( Setting::get_main_setting_by_code('width_height')->value ?? false );
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));


        $extra_inputs = (new General)->get_extra_inputs(92 , $lang);
        $this->return['master_inputs']     = $extra_inputs['master'];
        $this->return['detail_inputs']     = $extra_inputs['detail'];

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('06-006','add');

        $insert = $request->only(['quotation_id', 'purchase_order_id', 'proforma_invoice_date', 'proforma_invoice_description',
                                  'customer_id', 'sales_agent_id', 'invoice_discount', 'discount_type', 'discount' , 'extra_inputs']);

        $year   = array_search(date("Y", strtotime($insert['proforma_invoice_date'])), $this->data['years']);
        if($year == ''){
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        if(!isset($request['items']) || !is_array($request['items']) || count($request['items']) < 1 ){
            return $this->sendError('Erro In Items');
        }

        if(isset($insert['quotation_id']) && $insert['quotation_id'] > 0 ){
            $check = DB::table('quotation_m')
                ->where('quotation_m_id' , $insert['quotation_id'] )
                ->where('quotation_status' , 3)
                ->first();

            if(!isset($check->quotation_m_id)){
                return $this->sendError('Erro In quotation ID');
            }
        }
        $codes = (new ProformaInvoiceHelper)->get_next_codes($insert['proforma_invoice_date'], $year, $this->data['current_branch']);

        $master_seed = [
            'branch_id'                    => $this->data['current_branch'],
            'finance_year_id'              => $year,
            'year_ser'                     => $codes['year_ser'],
            'month_ser'                    => $codes['month_ser'],
            'branch_ser'                   => $codes['branch_ser'],
            'extra_inputs'                 => json_encode($insert['extra_inputs'] ?? [] ),
            'daily_journal_m_id'           => 0,
            'quotation_id'                 => $insert['quotation_id'] ?? 0,
            'purchase_order_id'            => $insert['purchase_order_id'],
            'proforma_invoice_date'        => $insert['proforma_invoice_date'],
            'proforma_invoice_description' => $insert['proforma_invoice_description'],
            'customer_id'                  => $insert['customer_id'],
            'sales_agent_id'               => $insert['sales_agent_id'],
            'invoice_discount'             => $insert['invoice_discount'] ?? 0,
            'discount_type'                => $insert['discount_type'] ?? 0,
            'discount'                     => $insert['discount'] ?? 0,
            'add_user'                     => auth()->user()->id
        ];

        $validator = validator()->make($master_seed, ProformaInvoice::$master_rules);

        if($validator->fails()){
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

            $m_id = ProformaInvoice::add_proforma_invoice_m($master_seed);

            if (isset($insert['quotation_id']) && !is_null($insert['quotation_id'])){
                DB::table('quotation_m')
                    ->where('quotation_m_id', $insert['quotation_id'])
                    ->update([
                        'proforma_invoice_id' => $m_id,
                        'quotation_status'    => 4
                    ]);
            }


            $items  = (new ProformaInvoiceHelper)->prepare_details( $m_id , $master_seed['quotation_id'] , $request['items'], 0 );

            if(count($items) < 1 ){
                DB::rollback();
                return $this->sendError(trans('error.add_fails'), []);
            }


            foreach ($items as $item){

                $validator = validator()->make($item, ProformaInvoice::$details_rules);
                if($validator->fails())
                    return $this->sendError(trans('error.add_fails'), $validator->errors());

                unset($item['d_id']);
                $d_id   = ProformaInvoice::add_proforma_invoice_d($item);

            }
            $updateMasterValues   = (new ProformaInvoiceHelper)->update_master_values($m_id);
            $generateDailyJournal = (new ProformaInvoiceHelper)->generate_daily_journal($m_id);

        DB::commit();
        return $this->sendResponse($m_id, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('06-006','view');

        (new ProformaInvoiceHelper)->update_master_values($id);

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('06-006', $lang)->sub_title ?? '';
        $this->return['master']            = ProformaInvoice::get_proforma_invoice_m($id, $lang);
        $this->return['details']           = ProformaInvoice::get_proforma_invoice_d($id, $lang);
        $this->return['files']             = ProformaInvoice::get_proforma_invoice_files($id);
        // $this->return['quotations']        = Quotation::all_quotations(array('active' => 1, 'lang' => $lang));
        $this->return['customers']         = Customer::all_customers(array('active' => 1, 'lang' => $lang ,  'is_search' => 1));
        $this->return['sales_agents']      = SalesAgent::all_sales_agents(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang , 'is_search' => 1));
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));
        $this->return['taxes']             = Tax::all_taxes(array('active' => 1, 'lang' => $lang));
        $this->return['width_height']      = boolval( Setting::get_main_setting_by_code('width_height')->value ?? false );
        $this->return['currencies']        = Currency::all_currency(array('active'=>1, 'lang' => $lang));


        $this->return['master']->extra_inputs = json_decode($this->return['master']->extra_inputs);
        foreach ($this->return['details'] as $key => $value) {
            $this->return['details'][$key]->extra_inputs = json_decode($this->return['details'][$key]->extra_inputs);
        }

        $extra_inputs = (new General)->get_extra_inputs(92 , $lang);
        $this->return['master_inputs']     = $extra_inputs['master'];
        $this->return['detail_inputs']     = $extra_inputs['detail'];



        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('06-006','edit');

        $update     = $request->only([ 'purchase_order_id', 'proforma_invoice_date', 'proforma_invoice_description',
                                  'customer_id', 'sales_agent_id', 'invoice_discount', 'discount_type', 'discount' , 'extra_inputs']);
        $master     = ProformaInvoice::get_proforma_invoice_m( $id , $lang );
        $details    = ProformaInvoice::get_proforma_invoice_d( $id , $lang );
        $check      = (new ProformaInvoiceHelper)->can_edit($id);

        if(!$check[0]){
            return $this->sendError($check[1], [], 200);
        }

        if(!isset($request['items']) || !is_array($request['items']) ){
            return $this->sendError('Items key requerid', [], 200);
        }

        DB::beginTransaction();
            $update['extra_inputs'] = json_encode($update['extra_inputs'] ?? [] );
            ProformaInvoice::update_proforma_invoice_m($update, $id);

            DB::table('proforma_invoice_d')
                ->where('proforma_invoice_m_id' , $id )
                ->delete();

            $items  = (new ProformaInvoiceHelper)->prepare_details( $id , $request['quotation_id'] ?? 0 , $request['items'],0 );

            if(count($items) < 1 ){
                DB::rollback();
                return $this->sendError(trans('error.add_fails'), []);
            }

            foreach ($items as $item){

                $validator = validator()->make($item, ProformaInvoice::$details_rules);
                if($validator->fails())
                    return $this->sendError(trans('error.add_fails'), $validator->errors());

                unset($item['d_id']);
                $d_id   = ProformaInvoice::add_proforma_invoice_d($item);

            }


            (new ProformaInvoiceHelper)->update_master_values($id);
            (new ProformaInvoiceHelper)->generate_daily_journal($id);

        DB::commit();

        return $this->sendResponse($id, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('06-006','delete');

        $master = ProformaInvoice::get_proforma_invoice_m($id);

        if(!isset($master->proforma_invoice_m_id) || $master->branch_id != $this->data['current_branch'] || !(new ProformaInvoiceHelper)->can_delete($id)[0] ){
            return $this->sendError(trans("error.delete_fail"));
        }

        DB::beginTransaction();

            DB::table('quotation_m')
                ->where('proforma_invoice_id' , $id)
                ->update([
                    'proforma_invoice_id'   => 0 ,
                    'quotation_status'      => 3
                ]);

            ProformaInvoice::delete_proforma_invoice_m($id);

        DB::commit();

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $master = ProformaInvoice::find($id);
        if(!isset($master->proforma_invoice_m_id) || $master->branch_id != $this->data['current_branch'] || !(new ProformaInvoiceHelper)->can_delete($id)[0] ){
            return $this->sendError(trans("error.delete_fail"));
        }
        DB::beginTransaction();

            if($master->proforma_invoice_active){
                $this->checkPermission('06-006','delete');
                DB::table('quotation_m')
                    ->where('proforma_invoice_id' , $id)
                    ->update([
                        'proforma_invoice_id'   => 0 ,
                        'quotation_status'      => 3
                    ]);

                ProformaInvoice::delete_proforma_invoice_m($id);
                $this->return['active'] = false;
            }else{
                $this->checkPermission('06-006','edit');

                DB::table('quotation_m')
                    ->where('quotation_m_id' , $master->quotation_id )
                    ->update([
                        'proforma_invoice_id'   => $id ,
                        'quotation_status'      => 4
                    ]);

                ProformaInvoice::update_proforma_invoice_m(array('proforma_invoice_active' => 1), $id);
                $this->return['active'] = true;
            }

        DB::commit();

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function uploadFiles($lang, Request $request)
    {
        if (isset($request['files']) && is_array($request['files']) && count($request['files']) > 0){
            if ($request->hasFile('files')){
                foreach ($request->file('files') as $file){
                    $upload = (new Upload)->uploadBase64File($file, 'proforma_invoice_files');
                    if(!$upload)
                        return $this->sendError(trans("error.Something Error"));
                    DB::table('proforma_invoice_files')->insert([
                        'proforma_invoice_m_id' => $request['id'],
                        'proforma_invoice_d_id' => 0,
                        'file_name'             => $upload['filename'],
                        'file_path'             => $upload['url'],
                        'file_extension'        => $upload['extension'],
                    ]);
                }
            }
        }elseif (isset($request['files']) && !is_array($request['files']) && strlen($request['files']) > 0){
            $upload = (new Upload)->uploadBase64File($request['files'], 'proforma_invoice_files');
            if(!$upload)
                return $this->sendError(trans("error.Something Error"));
            DB::table('proforma_invoice_files')->insert([
                'proforma_invoice_m_id' => $request['id'],
                'proforma_invoice_d_id' => 0,
                'file_name'             => $upload['filename'],
                'file_path'             => $upload['url'],
                'file_extension'        => $upload['extension'],
            ]);
        }

        return $this->sendResponse([], trans('main.Upload Successfully'));
    }

    public function get_customer_quotations($lang, Request $request)
    {
        $this->return['items'] = DB::table('quotation_m')
            ->where('customer_id', $request['customer_id'])
            ->where('quotation_status', 3)
            ->selectRaw('quotation_m_id as id , quotation_date , total_with_vat')
            ->get();

        return $this->sendResponse($this->return, trans('main.Data Retrieved Successfully'));
    }

    public function get_quotation_data($lang, Request $request)
    {
        $this->return['master'] = Quotation::get_quotation_m($request['quotation_id'], $lang);
        $this->return['items']  = Quotation::get_quotation_d($request['quotation_id'], $lang);

        return $this->sendResponse($this->return, trans('main.Data Retrieved Successfully'));
    }

    public function print_invoice($id, Request $request)
    {
        $input = $request->all();


        $this->data['lang']        = ($request['clientID'] == '1001'? 'en' : 'ar' );
        $this->data['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('06-006' , $this->data['lang']);
        $this->data['master']      = ProformaInvoice::get_proforma_invoice_m($id, $this->data['lang']);
        $this->data['details']     = ProformaInvoice::get_proforma_invoice_d($id, $this->data['lang']);
        $this->data['customer']    = Customer::get_customer($this->data['master']->customer_id, $this->data['lang']);
        $this->data['logo']        = get_logo($this->data['master']->branch_id);
        $this->data['branch_data'] = Branch::get_branch($this->data['master']->branch_id);
        $this->data['master']->tafqeet = ( new General )->numberToText($this->data['lang'], $this->data['master']->total_with_vat , $this->data['customer']->currency_code , $this->data['customer']->big_sign , $this->data['customer']->small_sign );


        $temp_name                 = $input['print_temp'] ?? Setting::get_main_setting_by_code('proforma_invoice_print')->value ?? 'temp_1';
        $print_temp                = 'Print.Sales.proforma_invoice.'.$temp_name;

        $this->data['qrcode']      = GenerateQrCode::fromArray([
            new Seller($this->data['branch_data']->vat_register_name), // Seller Name
            new TaxNumber($this->data['branch_data']->vat_number), // Seller Tax Number
            new InvoiceDate(date('Y-m-d\TH:i:s', strtotime($this->data['master']->add_date))),
            new InvoiceTotalAmount($this->data['master']->total_with_vat), // Invoice Total Amount
            new InvoiceTaxAmount(round($this->data['master']->total_vat, 2)) // Invoice Tax Amount
        ])->render();

        switch (@$input['type']) {
            case 'excel':
                $this->data['logo'] = public_path().$this->data['logo'];
                $export = new InvoicesExport($this->data, $print_temp);
                return Excel::download($export, 'balances.xlsx');
                break;
            case 'pdf':
                $this->data['logo'] = public_path().$this->data['logo'];
                $pdf = PDF::loadView($print_temp, array('data' => $this->data ));
                return $pdf->stream();
                break;
            case 'image':
                $this->data['logo'] = public_path().$this->data['logo'];
                $image = App::make('snappy.image.wrapper');
                $image->loadView($print_temp, array('data' => $this->data));
                return $image->stream();
                break;
            case 'print':
                return view($print_temp)->with('data', $this->data);
                break;
        }
    }
}
