<?php
namespace App\Http\Controllers\API\V1\Main;
use App\Http\Controllers\API\V1\BaseController;
use App\Classes\General\Upload;
use App\Models\Barbers\Barber;
use App\Models\Main\Branch;
use App\Models\Main\FinanceYear;
use App\Models\User;
use App\Models\Main\ScreenSub;
use App\Models\Main\SalesAgent;
use Illuminate\Http\Request;
use Auth;
use Illuminate\Support\Facades\App;
use Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class UserController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('01-008', 'view');

        $this->filter = array(
            'user_active' => $request['active'] ?? 1 ,
            'rows'        => $request['rows'] ?? null ,
            'page'        => $request['page'] ?? 1 ,
            'word'        => $request['word'] ?? null ,
            'lang'        => $lang,
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('01-008', $lang)->sub_title ?? '';
        $this->return['items']       = User::all_users($this->filter);

        return $this->sendResponse($this->return);

    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('01-008','add');


        $this->return['titles']          = DB::table('title')->select('title_id as id', 'title_name_'.$lang.' as name')->get();
        $this->return['all_permissions'] = ScreenSub::all_screen_subs($lang);
        $this->return['screen_name']     = ScreenSub::get_screen_sub_by_screen_code('01-008', $lang);
        $this->return['sales_agents']    = SalesAgent::all_sales_agents( $lang );
        $this->return['barbers']         = Barber::get_barber_lookup($lang);
        $this->return['branchs']         = Branch::all_branchs( array('lang' => $lang, 'active' => 1 ));
        $this->return['financial_years'] = FinanceYear::all_finance_years(array('lang' => $lang, 'active' => 1));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        App::setLocale($lang);
        $this->checkPermission('01-008', 'add');


        $validator = validator()->make($request->all(), User::$rules);

        if ($validator->fails()) {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        $seed = array(
            'user_full_name' => $request['user_full_name'],
            'username'       => $request['username'],
            'email'          => $request['email'],
            'mobile'         => $request['mobile'],
            'password'       => Hash::make($request['password']),
            'title_id'       => $request['title_id'],
            'sales_agent_id' => $request['sales_agent_id'] ?? 1,
            'default_branch' => $request['default_branch'] ?? 0,
            'view_all_data'  => $request['view_all_data'] ?? 1,

            'barber_id'      => $request['barber_id'] ?? 0,
            'default_year'   => $request['default_year'] ?? 0,
            'pos'            => $request['pos'] ?? 0,
            'user_active'    => 1,
            'user_titles'    => (is_array($request['user_titles'] ?? '')) ? implode(',', $request['user_titles']) : $request['user_titles'],
        );

        // check if the title_id exists in user_titles
        $title_id = $request['title_id'];
        $user_titles_input = $request['user_titles'];
        $user_titles_array = is_array($user_titles_input) ? $user_titles_input : explode(',', $user_titles_input);

        // Check if title_id exists in user_titles array
        if (!in_array($title_id, $user_titles_array)) {
            return $this->sendError(trans('main.the job title should be exists in the responsibilities'), []);
        }

        

        if (isset($request['image_url']) && strlen($request['image_url']) > 100) {
            $upload = new Upload;
            $check  = $upload->uploadBase64Image($request['image_url'], 'profile_images', 300, 300);
            if(!$check['check'])
                return $this->sendError(trans('error.add_fails'), $check['msg']);
            else
                $seed['image_url'] = $check['url'];
        }

        DB::beginTransaction();

        $id = User::add_user($seed);

        if (isset($request['user_titles']) && !is_null($request['user_titles']) && is_array($request['user_titles'])){
            $titles = DB::table('title_permissions')->whereIn('title_id', $request['user_titles'])->get();
        }else{
            $titles = DB::table('title_permissions')->where('title_id', $request['user_titles'])->get();
        }

        if (isset($titles)){
            foreach ($titles as $value) {
                DB::table('user_permissions')->insert([
                    'user_id'       => $id ,
                    'title_id'      => $value->title_id ,
                    'screen_sub_id' => $value->screen_sub_id ,
                    'permission'    => $value->permission,
                    'add_user'      => auth()->user()->id ,
                    'add_date'      => date("Y-m-d H:i:s")
                ]);
            }
        }

        $branch = Branch::get_branch($seed['default_branch']);

        if(strlen($branch->users) <= 0){
            $users = $seed['default_branch'];
        }else{
            $users = $branch->users.",".$id;
        }

        Branch::update_branch(['users' => $users], $seed['default_branch']);

        $year = FinanceYear::get_finance_year($seed['default_year']);

        if(strlen($year->users) <= 0){
            $users = $seed['default_year'];
        }else{
            $users = $year->users.",".$id;
        }

        FinanceYear::update_finance_year(['users' => $users], $seed['default_year']);

        DB::commit();
        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id, Request $request)
    {
        $this->checkPermission('01-008','view');

        $this->return['screen_name']          = ScreenSub::get_screen_sub_by_screen_code('01-008', $lang)->sub_title ?? '';
        $this->return['data']                 = User::find($id);
        $this->return['barbers']              = Barber::get_barber_lookup($lang);
        $this->return['branchs']              = Branch::all_branchs(array('lang' => $lang));
        $this->return['financial_years']      = FinanceYear::all_finance_years(array('lang' => $lang, 'active' => 1));
        $this->return['titles']               = DB::table('title')->select('title_id as id', 'title_name_'.$lang.' as name')->get();
        $this->return['all_permissions']      = ScreenSub::all_screen_subs(array('lang' => $lang));
        $user_permissions                     = User::get_user_permission($id, $lang);
        foreach ($user_permissions as $key => $value) {
            $user_permissions[$key]->permission = json_decode( $value->permission );
        }

        $this->return['user_permissions']  = collect($user_permissions)->groupBy('title_id')->toArray();

        $user_permissions                  = User::get_user_permission($id,null, true)->groupBy('screen_code')->toArray();

        foreach ($this->return['all_permissions'] as $key => $value) {
            $screens     = array('1' => 'view' , '2' => 'add' , '3' => 'edit' , '4' => 'delete' , '5' => 'print');
            $actions     = explode(',', $value->actions);
            $permissions = json_decode($user_permissions[$value->screen_code][0]->permission ?? "[]");
            $sync        = array();

            foreach($actions as $action_key => $row){
                if(in_array($row, $permissions)){
                    $sync[$screens[$row]] = true;
                }else{
                    $sync[$screens[$row]] = false;
                }
            }

            $this->return['all_permissions'][$key]->permissions = $sync;
        }

        $this->return['sales_agents'] = SalesAgent::all_sales_agents($lang);

        $this->return['all_my_years'] = DB::table('finance_years')
            ->whereRaw("find_in_set(".$id.", users)")
            ->pluck('name', 'finance_year_id')->toArray();

        $this->return['all_my_branchs'] = DB::table('branchs')
            ->whereRaw("find_in_set(".$id.", users)")
            ->pluck('name_'.$lang.' as name', 'branch_id')->toArray();

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        try{
        App::setLocale($lang);

        $this->checkPermission('01-008', 'edit');
        $user   = User::find($id);
        $rules  = User::$rules;
        $rules['username'] = $rules['username'] . ',id,' . $id;
        $rules['email']    = $rules['email'] . ',id,' . $id;
        $rules['mobile']   = $rules['mobile'] . ',id,' . $id;
        $attributes = User::attributeLabels();

        $update['user_full_name'] = $request['user_full_name'] ?? $user->user_full_name;
        $update['email']          = $request['email'] ?? $user->email;
        $update['mobile']         = $request['mobile'] ?? $user->mobile;
        $update['username']       = $request['username'] ?? $user->username;
        $update['title_id']       = $request['title_id'] ?? $user->title_id;
        $update['sales_agent_id'] = $request['sales_agent_id'] ?? $user->sales_agent_id;
        $update['default_branch'] = $request['default_branch'] ?? $user->default_branch;
        $update['view_all_data']  = $request['view_all_data'] ?? $user->view_all_data;

        $update['default_year']   = $request['default_year'] ?? $user->default_year;
        $update['barber_id']      = $request['barber_id'] ?? $user->barber_id;
        $update['pos']            = $request['pos'] ?? $user->pos;
        $update['user_titles']    = is_array($request['user_titles'] ?? '') ? implode(',', $request['user_titles']): ($request['user_titles'] ?? $user->user_titles);
        $update['user_active']    = $request['user_active'] ?? $user->user_active;
        $update['password']       = $user->password ?? $request['password'];
        $update['title_id']       = $request['title_id'] ?? $user->title_id;



        if (isset($request['password']) && strlen($request['password']) > 0) {
            $update['password'] = Hash::make($request['password']);
        }

            
        $validator = validator()->make($update, $rules,[], $attributes);

        if ($validator->fails()) {
            return $this->sendError($validator->errors()->all(), []);
        }

        // check if the title_id exists in user_titles
        $title_id = $request['title_id'] ?? $user->title_id;
        $user_titles_input = $request['user_titles'] ?? $user->user_titles;
        $user_titles_array = is_array($user_titles_input) ? $user_titles_input : explode(',', $user_titles_input);

        // Check if title_id exists in user_titles array
        if (!in_array($title_id, $user_titles_array)) {
            return $this->sendError(trans('main.the job title should be exists in the responsibilities'), []);
        }

        if(isset($request['image_url']) && strlen($request['image_url']) > 100){
            $upload = new Upload;
            $check  = $upload->uploadBase64Image($request['image_url'], 'profile_images', 300, 300);
            if(!$check['check'])
                return $this->sendError(trans('error.add_fails'), $check['msg']);
            else
                $update['image_url'] = $check['url'];
        }

        if (isset($request['user_titles']) && is_array($request['user_titles'])){

            DB::table('user_permissions')
                ->where('user_id', $id)
                ->delete();

            $titles = DB::table('title_permissions')
                ->whereIn('title_id', $request['user_titles'])
                ->get();

            if (isset($titles)){
                foreach ($titles as $value) {
                    DB::table('user_permissions')->insert([
                        'user_id'       => $id ,
                        'title_id'      => $value->title_id ,
                        'screen_sub_id' => $value->screen_sub_id ,
                        'permission'    => $value->permission,
                        'add_user'      => auth()->user()->id ,
                        'add_date'      => date("Y-m-d H:i:s")
                    ]);
                }
            }
        }

        if(count( $update ) > 0){
            User::update_user($update, $id);
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
        } catch (\Exception $e) {
            return $this->sendError($e->getMessage(), []);
        }
    }

    public function destroy($lang, $id)
    {
        App::setLocale($lang);
        $this->checkPermission('01-008','delete');


        User::delete_user($id);
        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function change_password($lang, Request $request)
    {
        $user = DB::table('users')->where('id' , auth()->guard('api')->user()->id)->first();

        if(!Hash::check($request['old_password'], $user->password)){
            return $this->sendError('Old Password Error', []);
        }

        if($request['new_password'] !== $request['new_password2']){
            return $this->sendError('New Password not match', []);
        }

        User::update_user(array('password' => Hash::make($request['new_password'])), auth()->user()->id);

        return $this->sendResponse([], trans('main.update_success'));
    }

    public function update_permission($lang, $id, Request $request)
    {
        App::setLocale($lang);
        
        // $this->checkPermission('01-008', 'edit');

        if (!isset($request['data']) || !is_array($request['data']) || count($request['data']) < 1) {
            return $this->sendError('data array is required', []);
        }

        DB::beginTransaction();
        $empty_permissions = true; // assume all are empty
        
        foreach ($request['data'] as $key => $value) {
            // Validate each item in the data array has the required keys
            if (!isset($value['sub_id']) || !isset($value['title_id']) || !isset($value['permissions'])) {
                return $this->sendError( trans('main.You must select at least one permission for the selected job title') , [] );
            }

            // Check if permissions is a string and convert it to an array
            if (!is_array($value['permissions'])) {
                $value['permissions'] = explode(',', $value['permissions']);
            }

            // Check if sub_id is valid
            $screen_sub = ScreenSub::where('screen_sub_id', $value['sub_id'])->first();
            if (!$screen_sub) {
                return $this->sendError('Invalid sub_id: ' . $value['sub_id'], []);
            }

            // Check if title_id is valid
            $title = DB::table('title')->where('title_id', $value['title_id'])->first();
            if (!$title) {
                return $this->sendError('Invalid title_id: ' . $value['title_id'], []);
            }

            // Check if at least one permissions array is NOT empty
            if (count($value['permissions']) > 0) {
                $empty_permissions = false;
            }
            
            DB::table('user_permissions')
                ->where('user_id', $id)
                ->where('title_id', $value['title_id'])
                ->where('screen_sub_id', $value['sub_id'])
                ->delete();

            User::insert_user_permission(
                array(
                    'screen_sub_id' => $value['sub_id'] ,
                    'user_id'       => $id ,
                    'title_id'      => $value['title_id'],
                    'permission'    => json_encode($value['permissions']) ,
                    'add_user'      => auth()->user()->id ,
                    'add_date'      => date("Y-m-d H:i:s")
                )
            );
        }
        
        // Check if at least one permission was provided
        if ($empty_permissions) {
            return $this->sendError(trans('main.You must select at least one permission for the selected job title'), []);
        }

        DB::commit();
        return $this->sendResponse([], trans('main.update_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = User::find($id);

        if($item->user_active){
            $this->checkPermission('01-008','delete');

            User::delete_user($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('01-008','edit');

            User::update_user(array('user_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }

    public function change_user_defaults($lang, Request $request)
    {
        if (isset($request['branch_id']) && isset($request['year_id']) && isset($request['title_id'])){
            DB::table('users')
                ->where('id', auth()->guard('api')->user()->id)
                ->update([
                'default_branch' => $request['branch_id'],
                'default_year'   => $request['year_id'],
                'title_id'       => $request['title_id'],
            ]);
        }

        return $this->sendResponse([], trans('User Defaults Data Changed Successfully'));
    }

}
