<?php

namespace App\Http\Controllers\API\V1\Inventory;

use App\Classes\Inventory\StockInitialBalanceHelper;
use App\Classes\ItemStock;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Inventory\Item;
use App\Models\Inventory\StockInitialBalance;
use App\Models\Inventory\UnitMeasurement;
use App\Models\Inventory\Warehouse;
use App\Models\Main\ScreenSub;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StockInitialBalanceController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('03-006', 'view');

        $this->filter = array(
            'lang'         => $lang,
            'active'       => $request['active'] ?? 1,
            'rows'         => $request['rows'] ?? 20,
            'page'         => $request['page'] ?? 1,
            'word'         => $request['word'] ?? null,
            'branch'       => $this->data['current_branch'],
            'finance_year' => $this->data['current_year']
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('03-006', $lang)->sub_title ?? '';
        $this->return['items']       = StockInitialBalance::all_stock_initial_balances($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('03-006', 'add');

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('03-006', $lang)->sub_title ?? '';
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang, 'is_service' => 0));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('03-006','add');

        $insert = $request->only(
            ['stock_initial_balance_date', 'stock_initial_balance_description', 'warehouse_id', 'items']
        );

        $year   = array_search(date("Y", strtotime($insert['stock_initial_balance_date'])), $this->data['years']);

        if ($year == '') {
            return $this->sendError(trans("error.finance_year_not_aval"));
        }

        $codes = (new StockInitialBalanceHelper)->get_next_codes($insert['stock_initial_balance_date'], $year, $this->data['current_branch']);

        $master_seed = [
            'branch_id'                         => $this->data['current_branch'],
            'finance_year_id'                   => $year,
            'year_ser'                          => $codes['year_ser'],
            'month_ser'                         => $codes['month_ser'],
            'branch_ser'                        => $codes['branch_ser'],
            'daily_journal_m_id'                => 0,
            'stock_initial_balance_date'        => $insert['stock_initial_balance_date'],
            'stock_initial_balance_description' => $insert['stock_initial_balance_description'],
            'warehouse_id'                      => $insert['warehouse_id'],
            'add_user'                          => auth()->user()->id
        ];

        $validator = validator()->make($master_seed, StockInitialBalance::$master_rules);

        if ($validator->fails()) {
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

        $m_id = StockInitialBalance::add_stock_initial_balance_m($master_seed);
        $source = ItemStock::First_Stocktaking_source; // + stock
        if (isset($insert['items']) && is_array($insert['items'])) {
            foreach ($insert['items'] as $item) {
                $selectedItem = DB::table('items')->where('item_id', $item['item_id'])->first();
                if (isset($item['unit_id']) && !is_null($item['unit_id'])) {
                    if ($item['unit_id'] == $selectedItem->big_unit_id) {
                        $item['unit_type'] = 3;
                    } elseif ($item['unit_id'] == $selectedItem->mid_unit_id) {
                        $item['unit_type'] = 2;
                    } elseif ($item['unit_id'] == $selectedItem->small_unit_id) {
                        $item['unit_type'] = 1;
                    } else {
                        $item['unit_type'] = 3;
                    }
                }
                $details_seed = array(
                    'stock_initial_balance_m_id' => $m_id,
                    'item_id'                    => $item['item_id'],
                    'unit_id'                    => $item['unit_id'],
                    'unit_type'                  => $item['unit_type'] ?? 0,
                    'quantity'                   => $item['quantity'] ?? 0,
                    'price'                      => $item['price'] ?? 0,
                    'total_price'                => ($item['price'] * $item['quantity']),
                    'add_user'                   => auth()->user()->id
                );

                $validator = validator()->make($details_seed, StockInitialBalance::$details_rules);

                if ($validator->fails()) {
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }

                $d_id = StockInitialBalance::add_stock_initial_balance_d($details_seed);
                $warehouse_id = $insert['warehouse_id'];
                // Item Stock
                (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $warehouse_id, $source, $m_id, $d_id, $item['quantity'], 0, $item['unit_id'], $item['unit_type'], 0);
            }
        }

        (new StockInitialBalanceHelper)->update_master_values($m_id);
        (new StockInitialBalanceHelper)->generate_daily_journal($m_id);

        DB::commit();
        return $this->sendResponse([], trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('03-006', 'view');

        $this->return['screen_name']       = ScreenSub::get_screen_sub_by_screen_code('03-006', $lang)->sub_title ?? '';
        $this->return['master']            = StockInitialBalance::get_stock_initial_balance_m($id, $lang);
        $this->return['details']           = StockInitialBalance::get_stock_initial_balance_d($id, $lang);
        $this->return['warehouses']        = Warehouse::all_warehouses(array('active' => 1, 'lang' => $lang));
        $this->return['items']             = Item::all_items(array('active' => 1, 'lang' => $lang, 'is_service' => 0));
        $this->return['unit_measurements'] = UnitMeasurement::all_unit_measurements(array('active' => 1, 'lang' => $lang));

        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('03-006', 'edit');

        $inputs = $request->all();
        $update_m = [
            'stock_initial_balance_description' => $inputs['stock_initial_balance_description'],
            'stock_initial_balance_date'        => $inputs['stock_initial_balance_date']
        ];
        DB::beginTransaction();
        $master    = StockInitialBalance::get_stock_initial_balance_m($id, $lang);

        StockInitialBalance::update_stock_initial_balance_m($update_m, $id);

        $new_details = collect($inputs['items'])->pluck('d_id')->toArray();
        $details = DB::table('stock_initial_balance_d')->where('stock_initial_balance_m_id', $id)->pluck('stock_initial_balance_d_id')->toArray();
        foreach ($details as $detail) {

            if (!in_array($detail, $new_details)) {
                DB::table('stock_initial_balance_d')->where('stock_initial_balance_d_id', $detail)->delete();
            }
        }
        $source = ItemStock::First_Stocktaking_source; // + stock

        foreach ($inputs['items'] as $item) {
            if ($item['d_id'] == 0) {
                $insert_d = [
                    'item_id'                   => $item['item_id'],
                    'unit_id'                   => $item['unit_id'],
                    'stock_initial_balance_m_id' => $id,
                    'quantity'                  => $item['quantity'],
                    'price'                     => $item['price'],
                    'total_price'               => ($item['price'] * $item['quantity']),
                    'add_user'                  => auth()->user()->id,
                ];
                $item['d_id'] = StockInitialBalance::add_stock_initial_balance_d($insert_d);
            } else {
                $update_d = [
                    'item_id'                   => $item['item_id'],
                    'unit_id'                   => $item['unit_id'],
                    'stock_initial_balance_m_id' => $id,
                    'quantity'                  => $item['quantity'],
                    'price'                     => $item['price'],
                    'total_price'               => ($item['price'] * $item['quantity']),
                ];
                StockInitialBalance::update_stock_initial_balance_d($update_d, $item['d_id']);
            }

            DB::table('stock_logs')
                ->where('branch_id', $master->branch_id)
                ->where('item_id', $item['item_id'])
                ->where('warehouse_id', $master->warehouse_id)
                ->where('source', 'fst')
                ->where('source_m_id', $id)
                ->where('source_d_id', $item['d_id'])
                ->where('unit_id', $item['unit_id'])
                ->delete();

            (new ItemStock)->update_item_stock($this->data['current_branch'], $item['item_id'], $master->warehouse_id, $source,$id, $item['d_id'], $item['quantity'], 0, $item['unit_id'], $item['unit_type'], 0);

        }
        DB::commit();

        return $this->sendResponse($this->return, trans('main.update_success'));
    }


    public function destroy($lang, $id)
    {
        $this->checkPermission('03-006', 'delete');

        $master = StockInitialBalance::get_stock_initial_balance_m($id);

        if (!isset($master->stock_initial_balance_m_id) || $master->branch_id != $this->data['current_branch'] || !(new StockInitialBalanceHelper)->can_edit_or_delete($id)) {
            return $this->sendError(trans("error.delete_fail"));
        }

        StockInitialBalance::delete_stock_initial_balance_m($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = StockInitialBalance::find($id);

        if ($item->stock_initial_balance_active) {
            $this->checkPermission('03-006', 'delete');

            if (!(new StockInitialBalanceHelper)->can_edit_or_delete($id)) {
                return $this->sendError(trans('error.delete_fails'), []);
            }

            StockInitialBalance::delete_stock_initial_balance_m($id);
            $this->return['active'] = false;
        } else {
            $this->checkPermission('03-006', 'edit');

            if (!(new StockInitialBalanceHelper)->can_edit_or_delete($id)) {
                return $this->sendError(trans('error.update_fails'), []);
            }

            StockInitialBalance::update_stock_initial_balance_m(array('stock_initial_balance_active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
