<?php

namespace App\Http\Controllers\API\V1\Elevators;
use App\Http\Controllers\API\V1\BaseController;
use App\Models\Elevators\ElevatorModels;
use App\Models\Elevators\ElevatorTypes;
use App\Models\Main\ScreenSub;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class ElevatorTypesController extends BaseController
{
    protected $return = array();
    protected $filter = array();

    public function index($lang, Request $request)
    {
        $this->checkPermission('12-001','view');

        $this->filter = array(
            'lang'   => $lang ,
            'active' => $request['active'] ?? 1 ,
            'rows'   => $request['rows'] ?? 20 ,
            'page'   => $request['page'] ?? 1 ,
            'word'   => $request['word'] ?? null ,
        );

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('12-001', $lang)->sub_title ?? '';
        $this->return['items']       = ElevatorTypes::all_types($this->filter);

        return $this->sendResponse($this->return);
    }

    public function create($lang, Request $request)
    {
        $this->checkPermission('12-001','add');

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('12-001', $lang)->sub_title ?? '';

        return $this->sendResponse($this->return);
    }

    public function store($lang, Request $request)
    {
        $this->checkPermission('12-001','add');

        $inputs = $request->only(['name_en', 'name_ar']);

        $master_seed = [
            'name_en'  => $inputs['name_en'],
            'name_ar'  => $inputs['name_ar'],
        ];

        $validator = validator()->make($master_seed, ElevatorTypes::$master_rules);

        if($validator->fails()){
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

        $this->return['id'] = ElevatorTypes::add_type($master_seed);

        DB::commit();
        return $this->sendResponse($this->return, trans('main.add_success'));
    }

    public function show($lang, $id)
    {
        $this->checkPermission('12-001','view');

        $this->return['screen_name'] = ScreenSub::get_screen_sub_by_screen_code('12-001', $lang)->sub_title ?? '';
        $this->return['master']      = ElevatorTypes::get_type($id, $lang);
        $this->return['models']      = ElevatorModels::all_models(['type_id'=>$id, 'lang'=>$lang , 'active'=>1]);

        return $this->sendResponse($this->return);
    }

    public function update($lang, $id, Request $request)
    {
        $this->checkPermission('12-001','edit');

        $inputs = $request->all();

        $master_seed = [
            'name_en'  => $inputs['type_name_en'],
            'name_ar'  => $inputs['type_name_ar'],
        ];
        $update_rules = [
            'name_en' => [
                'required',
                'string',
                Rule::unique('elevators_types')->ignore($id,'type_id'),
            ],
            'name_ar' => [
                'required',
                'string',
                Rule::unique('elevators_types')->ignore($id,'type_id'),
            ]
        ];
        $validator = validator()->make($master_seed, $update_rules);

        if($validator->fails()){
            return $this->sendError(trans('error.add_fails'), $validator->errors());
        }

        DB::beginTransaction();

        ElevatorTypes::update_type($master_seed, $id);
        if (isset($inputs['models']) && is_array($inputs['models'])){
            DB::table('elevators_models_m')->where('elevator_type_id',$id)->delete();
            foreach ($inputs['models'] as $model){
                $master_seed = [
                    'name_en'           => $model['name_en'],
                    'name_ar'           => $model['name_ar'],
                    'elevator_type_id'  => $id,
                ];

                $validator = validator()->make($master_seed, ElevatorModels::$master_rules);

                if($validator->fails()){
                    return $this->sendError(trans('error.add_fails'), $validator->errors());
                }
                $m_id = ElevatorModels::add_model_m($master_seed);
                if (isset($model['items']) && is_array($model['items'])){
                    DB::table('elevators_models_d')->where('model_m_id',$m_id)->delete();
                    foreach ($model['items'] as $item){
                        $details_seed = array(
                            'model_m_id'        => $m_id,
                            'item_name'         => $item['item_name'],
                            'description'       => $item['description'],
                        );

                        $validator = validator()->make($details_seed, ElevatorModels::$details_rules);

                        if($validator->fails()){
                            return $this->sendError(trans('error.add_fails'), $validator->errors());
                        }
                        ElevatorModels::add_model_d($details_seed);
                    }
                }
            }
        }
        DB::commit();
        return $this->sendResponse($id, trans('main.update_success'));
    }

    public function destroy($lang, $id)
    {
        $this->checkPermission('12-001','delete');

        ElevatorTypes::delete_type($id);

        return $this->sendResponse([], trans('main.delete_success'));
    }

    public function toggle_active($lang, $id)
    {
        $item = ElevatorTypes::find($id);

        if($item->active){
            $this->checkPermission('12-001','view');
            $check = DB::table('elevators_quotations')->where('type_id',$item->elevator_type_id)->count();
            if($check > 0){
                $this->sendError(trans('error.cannot_delete_type_in_use'));
            }
            ElevatorTypes::delete_type($id);
            $this->return['active'] = false;
        }else{
            $this->checkPermission('12-001','view');

            ElevatorTypes::update_type(array('active' => 1), $id);
            $this->return['active'] = true;
        }

        return $this->sendResponse($this->return, trans('main.update_success'));
    }
}
